implementation module StdIOCommon


//	Clean Object I/O library, version 1.1

//	Common types for the event I/O system and their access-rules.


import	StdInt, StdOverloaded, StdBool, StdString
import	StdIOBasic
from	id			import	Id, WindowMenuId, toId, RId, R2Id, RIdtoId, R2IdtoId, toString
from	key			import	SpecialKey, 
								BeginKey, 
								ClearKey, 
								DeleteKey, DownKey, 
								EndKey, EnterKey, EscapeKey, 
								F1Key,  F2Key,  F3Key,  F4Key,  F5Key,  F6Key, F7Key, F8Key, F9Key, 
								F10Key, F11Key, F12Key, F13Key, F14Key, F15Key, 
								HelpKey, 
								LeftKey, 
								PgDownKey, PgUpKey, 
								RightKey, 
								UpKey

/*	The SelectState type.								*/

::	SelectState		=	Able | Unable

instance == SelectState
where
	(==) :: !SelectState !SelectState -> Bool
	(==) Able	select = enabled select
	(==) Unable	select = not (enabled select)
instance ~ SelectState where
	(~) :: !SelectState -> SelectState
	(~) Able	= Unable
	(~) Unable	= Able

enabled :: !SelectState -> Bool
enabled Able	= True
enabled _		= False


/*	The MarkState type.									*/

::	MarkState		=	Mark | NoMark

instance == MarkState
where
	(==) :: !MarkState	!MarkState	-> Bool
	(==) Mark	mark = marked mark
	(==) NoMark	mark = not (marked mark)
instance ~ MarkState where
	(~) :: !MarkState -> MarkState
	(~) Mark	= NoMark
	(~) _		= Mark

marked :: !MarkState -> Bool
marked Mark   	= True
marked _		= False


/*	The KeyboardState type.								*/

::	KeyboardState
	=	CharKey		Char		KeyState			// ASCII character input
	|	SpecialKey	SpecialKey	KeyState Modifiers	// Special key input
::	KeyState
	=	KeyDown		IsRepeatKey						// Key is down
	|	KeyUp										// Key goes up
::	IsRepeatKey										// Flag on key down:
	:==	Bool										// True iff key is repeating
::	Key
	=	IsCharKey	 Char
	|	IsSpecialKey SpecialKey
::	KeyboardStateFilter								// Predicate on KeyboardState:
	:==	KeyboardState -> Bool						// evaluate KeyFunction only if predicate holds and SelectState is Able

getKeyboardStateKeyState :: !KeyboardState -> KeyState
getKeyboardStateKeyState (CharKey    _ kstate  ) = kstate
getKeyboardStateKeyState (SpecialKey _ kstate _) = kstate

getKeyboardStateKey :: !KeyboardState -> Key
getKeyboardStateKey (CharKey    char  _) = IsCharKey   char
getKeyboardStateKey (SpecialKey key _ _) = IsSpecialKey key

instance == KeyState
where
	(==) :: !KeyState !KeyState -> Bool
	(==) KeyUp				key	= case key of
									KeyUp				-> True
									_					-> False
	(==) (KeyDown repeat)	key	= case key of
									(KeyDown repeat`)	-> repeat==repeat`
									_					-> False


/*	The MouseState type.								*/

::	MouseState
	=	MouseMove	Point Modifiers				// Mouse is up		(position & modifiers)
	|	MouseDown	Point Modifiers Int			// Mouse goes down	(position & modifiers & nr double down)
	|	MouseDrag	Point Modifiers				// Mouse is down	(position & modifiers)
	|	MouseUp		Point Modifiers				// Mouse goes up	(position & modifiers)
::	ButtonState
 	=	ButtonStillUp							// MouseMove
 	|	ButtonDown								// MouseDown 1
	|	ButtonDoubleDown						//			 2
	|	ButtonTripleDown						//           >2
	|	ButtonStillDown							// MouseDrag
 	|	ButtonUp								// MouseUp
::	MouseStateFilter							// Predicate on MouseState:
	:==	MouseState -> Bool						// Evaluate MouseFunction only if predicate holds and SelectState is Able

getMouseStatePos :: !MouseState -> Point
getMouseStatePos (MouseMove pos _)			= pos
getMouseStatePos (MouseDown pos _ _)		= pos
getMouseStatePos (MouseDrag pos _)			= pos
getMouseStatePos (MouseUp   pos _)			= pos

getMouseStateModifiers :: !MouseState -> Modifiers
getMouseStateModifiers (MouseMove _ mods)	= mods
getMouseStateModifiers (MouseDown _ mods _)	= mods
getMouseStateModifiers (MouseDrag _ mods)	= mods
getMouseStateModifiers (MouseUp   _ mods)	= mods

getMouseStateButtonState:: !MouseState	-> ButtonState
getMouseStateButtonState (MouseMove _ _)	= ButtonStillUp
getMouseStateButtonState (MouseDown _ _ nr)	= if (nr==1) ButtonDown 
											 (if (nr==2) ButtonDoubleDown
											 			 ButtonTripleDown
											 )
getMouseStateButtonState (MouseDrag _ _)	= ButtonStillDown
getMouseStateButtonState (MouseUp   _ _)	= ButtonUp

instance == ButtonState
where
	(==) :: !ButtonState	!ButtonState					-> Bool
	(==) ButtonStillUp		button	= case button of
										ButtonStillUp		-> True
										_					-> False
	(==) ButtonDown			button	= case button of
										ButtonDown			-> True
										_					-> False
	(==) ButtonDoubleDown	button	= case button of
										ButtonDoubleDown	-> True
										_					-> False
	(==) ButtonTripleDown	button	= case button of
										ButtonTripleDown	-> True
										_					-> False
	(==) ButtonStillDown	button	= case button of
										ButtonStillDown		-> True
										_					-> False
	(==) ButtonUp			button	= case button of
										ButtonUp			-> True
										_					-> False


/*	The SliderState type.								*/

::	SliderState
	=	{	sliderMin	:: !Int
		,	sliderMax	:: !Int
		,	sliderThumb	:: !Int
		}

instance == SliderState									// Equality on SliderState
where
	(==) :: !SliderState !SliderState -> Bool
	(==) s1 s2 = s1.sliderMin==s2.sliderMin && s1.sliderMax==s2.sliderMax && s1.sliderThumb==s2.sliderThumb


/*	The UpdateState type.								*/

::	UpdateState
	=	{	oldFrame	:: !ViewFrame
		,	newFrame	:: !ViewFrame
		,	updArea		:: !UpdateArea
		}
::	ViewDomain			:==	Rectangle
::	ViewFrame			:==	Rectangle
::	UpdateArea			:==	[ViewFrame]


/*	Modifiers indicates the meta keys that have been pressed (True) or not (False).	*/

::	Modifiers
	=	{	shiftDown	:: !Bool
		,	optionDown	:: !Bool
		,	commandDown	:: !Bool
		,	controlDown	:: !Bool
		,	altDown		:: !Bool
		}

NoModifiers		:==	{shiftDown=False,optionDown=False,commandDown=False,controlDown=False,altDown=False}
ShiftOnly		:==	{NoModifiers & shiftDown	= True}
OptionOnly		:== {NoModifiers & optionDown	= True}
CommandOnly		:== {NoModifiers & commandDown	= True}
ControlOnly		:== {NoModifiers & controlDown	= True}
AltOnly			:==	{NoModifiers & altDown		= True}


/*	The layout language used for windows and controls.	*/

::	ItemPos
	:==	(	ItemLoc
		,	ItemOffset
		)
::	ItemLoc
	//	Absolute:
	=	Fix		Point
	//	Relative to corner:
	|	LeftTop
	|	RightTop
	|	LeftBottom
	|	RightBottom
	//	Relative in next line:
	|	Left
	|	Center
	|	Right
	//	Relative to other item:
	|	LeftOf	Id
	|	RightTo	Id
	|	Above	Id
	|	Below	Id
	//	Relative to previous item:
	|	LeftOfPrev
	|	RightToPrev
	|	AbovePrev
	|	BelowPrev
::	ItemOffset
	:==	Vector

instance == ItemLoc
where
	(==) :: !ItemLoc		!ItemLoc -> Bool
	(==) (Fix pos1)			itemLoc	= case itemLoc of
										(Fix pos2)		-> pos1==pos2
										_				-> False
	(==) LeftTop			itemLoc	= case itemLoc of
										LeftTop			-> True
										_				-> False
	(==) RightTop			itemLoc	= case itemLoc of
										RightTop		-> True
										_				-> False
	(==) LeftBottom			itemLoc	= case itemLoc of
										LeftBottom		-> True
										_				-> False
	(==) RightBottom		itemLoc	= case itemLoc of
										RightBottom		-> True
										_				-> False
	(==) Left				itemLoc	= case itemLoc of
										Left			-> True
										_				-> False
	(==) Center				itemLoc	= case itemLoc of
										Center			-> True
										_				-> False
	(==) Right				itemLoc	= case itemLoc of
										Right			-> True
										_				-> False
	(==) (LeftOf	id1)	itemLoc	= case itemLoc of
										LeftOf	id2		-> id1==id2
										_				-> False
	(==) (RightTo	id1)	itemLoc	= case itemLoc of
										RightTo	id2		-> id1==id2
										_				-> False
	(==) (Above		id1)	itemLoc	= case itemLoc of
										Above	id2		-> id1==id2
										_				-> False
	(==) (Below		id1)	itemLoc	= case itemLoc of
										Below	id2		-> id1==id2
										_				-> False
	(==) LeftOfPrev			itemLoc	= case itemLoc of
										LeftOfPrev		-> True
										_				-> False
	(==) RightToPrev		itemLoc	= case itemLoc of
										RightToPrev		-> True
										_				-> False
	(==) AbovePrev			itemLoc	= case itemLoc of
										AbovePrev		-> True
										_				-> False
	(==) BelowPrev			itemLoc	= case itemLoc of
										BelowPrev		-> True
										_				-> False


/*	The Direction type.									*/

::	Direction
	=	Horizontal
	|	Vertical

instance == Direction
where
	(==) :: !Direction !Direction -> Bool
	(==) Horizontal direction	= case direction of
									Horizontal	-> True
									_			-> False
	(==) Vertical	direction	= case direction of
									Vertical	-> True
									_			-> False


/*	Document interface type of interactive processes.	*/

::	DocumentInterface
	=	NDI														// No       Document Interface
	|	SDI														// Single   Document Interface
	|	MDI														// Multiple Document Interface

instance == DocumentInterface
where
	(==) :: !DocumentInterface !DocumentInterface -> Bool
	(==) NDI xdi	= case xdi of
						NDI	-> True
						_	-> False
	(==) SDI xdi	= case xdi of
						SDI	-> True
						_	-> False
	(==) MDI xdi	= case xdi of
						MDI	-> True
						_	-> False


/*	Process attributes.									*/

::	ProcessAttribute ps											// Default:
	=	ProcessWindowPos	ItemPos								// Platform dependent
	|	ProcessWindowSize	Size								// Platform dependent
	|	ProcessWindowResize	(ProcessWindowResizeFunction ps)	// Platform dependent
	|	ProcessHelp			(IOFunction ps)						// No Help  facility
	|	ProcessAbout		(IOFunction ps)						// No About facility
	|	ProcessActivate		(IOFunction ps)						// No action on activate
	|	ProcessDeactivate	(IOFunction ps)						// No action on deactivate
	|	ProcessClose		(IOFunction ps)						// Process is closed
	|	ProcessShareGUI											// Process does not share GUI of parent
//	Attributes for MDI processes only:
	|	ProcessNoWindowMenu										// Process has WindowMenu

::	ProcessWindowResizeFunction ps
	:==	Size													// Old ProcessWindow size
	 ->	Size													// New ProcessWindow size
	 ->	ps -> ps


/*	Frequently used function types.						*/

::	IOFunction			ps	:==						ps -> ps
::	ModsIOFunction		ps	:==	Modifiers		->	ps -> ps
::	MouseFunction		ps	:== MouseState		->	ps -> ps
::	KeyboardFunction	ps	:== KeyboardState	->	ps -> ps


/*	Common error report types.							*/

::	ErrorReport													// Usual cause:
	=	NoError													// No error
	|	ErrorViolateDI											// Violation against document interface kind
	|	ErrorIdsInUse											// Object definition contains Ids that are already in use
	|	ErrorUnknownObject										// Object can not be found
	|	ErrorNotifierOpen										// It was tried to open a second send notifier


instance == ErrorReport
where
	(==) :: !ErrorReport !ErrorReport -> Bool
	(==) NoError			error	= case error of
										NoError				-> True
										_					-> False
	(==) ErrorViolateDI		error	= case error of
										ErrorViolateDI		-> True
										_					-> False
	(==) ErrorIdsInUse		error	= case error of
										ErrorIdsInUse		-> True
										_					-> False
	(==) ErrorUnknownObject	error	= case error of
										ErrorUnknownObject	-> True
										_					-> False
	(==) ErrorNotifierOpen	error	= case error of
										ErrorNotifierOpen	-> True
										_					-> False

instance toString ErrorReport where
	toString :: !ErrorReport -> {#Char}
	toString NoError			= "NoError"
	toString ErrorViolateDI		= "ErrorViolateDI"
	toString ErrorIdsInUse		= "ErrorIdsInUse"
	toString ErrorUnknownObject	= "ErrorUnknownObject"
	toString ErrorNotifierOpen	= "ErrorNotifierOpen"


::	OkBool									// iff True, the operation was successful
	:==	Bool
